const js = $('#sortingtable_js');

/**
 * Handles the event for changing the check state of a check-box within a table.
 * If the check-box state for all table items is changed, the event is applied
 * for all check-boxes of the table. If the state of the check-box from an
 * arbitrary row is changed, the check-box for all items is unchecked. After
 * applying this UI handling, the call-back handler is called with the
 * information, if at least one check-box is checked.<br>
 * The check-boxes of a table must have an ID starting with the prefix (and for
 * example a row), the check-box for all items appends '_all' to the prefix for
 * the ID.
 *
 * @param all
 *            the flag, if the check-box for all items is checked
 * @param checkBoxIdPrefix
 *            the prefix for the check-boxes
 * @param callback
 *            the callback to execute reporting the information if at least one
 *            table item is checked
 */
function tableCheckBoxChecked(all, checkBoxIdPrefix, callback) {
    if (all) {
        // check all items
        $('input[id^=' + checkBoxIdPrefix + ']:not(#' + checkBoxIdPrefix + '_all)').each(function () {
            $(this).prop('checked', $('#' + checkBoxIdPrefix + '_all').prop('checked'));
        });
    } else {
        // uncheck 'all'
        $('#' + checkBoxIdPrefix + '_all').prop('checked', false);
    }

    callback($('input[id^=' + checkBoxIdPrefix + ']:not(#' + checkBoxIdPrefix + '_all):checked').length);
}

/**
 * Returns the ids of all selected check-boxes of the document having an ID starting with the given prefix.
 *
 * @param checkBoxIdPrefix the prefix of the check-boxes to load
 * @param suffix           if the checkbox ids have a suffix that should be removed, specify this suffix here
 * @returns the selected check-box-ids of the document as String, concatenated with ','
 */
function getSelectedIds(checkBoxIdPrefix, suffix = null) {
    let selectedIds = '';
    $('input[id^=' + checkBoxIdPrefix + ']:not(#' + checkBoxIdPrefix + '_all):checked').each(function () {
        if (selectedIds !== '') {
            selectedIds += ',';
        }
        selectedIds += $(this).attr('id').substring(checkBoxIdPrefix.length + 1);
    });
    if (suffix !== null) {
        selectedIds = selectedIds.replace(suffix,'');
    }
    return selectedIds;
}

/**
 * Returns the numeric ids (removing any non-digit characters) of all selected check-boxes of the document having an ID
 * starting with the given prefix.
 *
 * @param checkBoxIdPrefix
 *            the prefix of the check-boxes to load
 * @returns the numeric selected check-box-ids of the document as String, concatenated
 *          with ','
 */
function getSelectedNumericIds(checkBoxIdPrefix) {
    let selectedIds = '';
    $('input[id^=' + checkBoxIdPrefix + ']:not(#' + checkBoxIdPrefix + '_all):checked').each(function () {
        if (selectedIds !== '') {
            selectedIds += ',';
        }
        let id = $(this).attr('id').substring(checkBoxIdPrefix.length + 1);
        if (isNaN(id))
            id = id.match(/\d+/)[0];
        selectedIds += id
    });
    return selectedIds;
}

/**
 * Returns the ids of all check-boxes of the document having an ID starting with
 * the given prefix.
 *
 * @param checkBoxIdPrefix
 *            the prefix of the check-boxes to load
 * @returns the check-box-ids of the document as String, concatenated with ','
 */
function getAllIds(checkBoxIdPrefix) {
    let selectedIds = '';
    $('input[id^=' + checkBoxIdPrefix + ']:not(#' + checkBoxIdPrefix + '_all)').each(function () {
        if (selectedIds !== '') {
            selectedIds += ',';
        }
        selectedIds += $(this).attr('id').substring(checkBoxIdPrefix.length + 1);
    });
    return selectedIds;
}

$(document).ready(tableCheckboxSetup(js.data('table-id'), js.data('has-checkbox')));