const pagingsortingtable_js = $('#pagingsortingtable_js');
const localized_columns = JSON.parse(atob(pagingsortingtable_js.attr('data-localizedcolumns')));
const locked_columns = JSON.parse(atob(pagingsortingtable_js.attr('data-lockedcolumns')));
const lockedColumnMsg = pagingsortingtable_js.attr('data-lockedcolumnsmessage')
let currentColumnVisibility = $('#column_visibility').val();
let currentColumnOrder = $('#column_order').val();

/**
 * Makes sure that the value of the manual page size field stays within bounds.
 */
function enforcePageSizeLimits() {
    const inputField = $("#paging_custom_size_input");
    let val = inputField.val();
    if (isNaN(val)) {
        inputField.val(200);
    } else if (val < 1) {
        inputField.val(1);
    } else if (val > 1000) {
        inputField.val(1000);
    }
}

/**
 * Determine if the column with the given index is currently set as a visible column in the columns config modal.
 *
 * @param i {string} the unique index of the column (unaffected by column ordering)
 * @returns {boolean} true if the column is currently set as a visible column in the columns config modal
 */
function columnIsVisible(i) {
    let columns_visible_field = $('input#column_visibility');
    let columns_visible = columns_visible_field.val().split(',');
    return columns_visible[i] === "1";
}

/**
 * Determine if the column with the given index is currently set as a visible column in the columns config modal.
 *
 * @param i {string}        the unique index of the column (unaffected by column ordering)
 * @param visible {boolean} use true to set visible, use false to hide
 */
function setColumnVisibility(i, visible) {
    let value = (visible === true) ? "1" : "0";
    let columns_visible_field = $('input#column_visibility');
    let columns_visible = columns_visible_field.val().split(',');
    columns_visible[i] = value;
    columns_visible_field.val(columns_visible.join(','));
}

/**
 * A locked column cannot be removed by the user (remove button is disabled).
 * This function reads from the JS parameters which columns are locked and returns if the given column is locked.
 *
 * @param i {string}  the unique index of the column (unaffected by column ordering)
 * @returns {boolean} true if the column i is locked, false otherwise
 */
function columnIsLocked(i) {
    return locked_columns[i] === 1
}

/**
 * This is used to sort the available columns by their localized name.
 */
function compareOptions(i, j) {
    return localized_columns[i].localeCompare(localized_columns[j]);
}

/**
 * Clears the columns config modal content and populates it again from scratch using the column_order and column_visibility fields.
 *
 * @param updateUnused {boolean}  use false here if the "available columns" section of the modal does not need to be updated
 * @param reloadCurrent {boolean} use true to reload the values that are currently in the database into the modal
 */
function updateColumnsModal(updateUnused, reloadCurrent) {
    if (reloadCurrent) {
        $('#column_visibility').val(currentColumnVisibility);
        $('#column_order').val(currentColumnOrder);
    }

    const column_order_field = $('input#column_order');
    const columns_used = column_order_field.val().split(',').filter(function (n) {
        return columnIsVisible(n);
    });

    if (columns_used.length === 0) {
        $('#columns_used_title').hide();
    } else {
        $('#columns_used_title').show();
    }

    let display = $('#columns_used');
    display.empty();
    for (let i = 0; i < columns_used.length; i++) {
        let spanElement = '<span class="input-group-text">' + (i + 1) + '</span>';
        if (i + 1 < 10) {
            spanElement = '<span class="input-group-text bl-conversion-input-group">' + (i + 1) + '</span>';
        }
        let columnLocked = columnIsLocked(columns_used[i]);
        display.append($('<div class="input-group">'
            + spanElement
            + '<button id="' + columns_used[i] + '_up" class="up btn btn-secondary" type="button" ' + (i === 0 ? 'disabled' : '') + ' ><span class="fa fa-chevron-up" aria-hidden="true"></span></button>'
            + '<button id="' + columns_used[i] + '_down" class="down btn btn-secondary" type="button" ' + (i === columns_used.length - 1 ? 'disabled' : '') + ' ><span class="fa fa-chevron-down" aria-hidden="true"></span></button>'
            + '<span class="form-control">' + localized_columns[columns_used[i]] + '</span>'
            + (columnLocked ? '<span data-toggle="tooltip" data-placement="bottom" title="' + lockedColumnMsg + '">' : '')
            + '<button id="' + columns_used[i] + '_remove" class="remove btn btn-danger" type="button" ' + (columnLocked ? 'disabled' : '') + ' ><span class="fa fa-remove"></span></button>'
            + (columnLocked ? '</span>' : '')
            + '</div>'));
    }

    if (updateUnused) {
        const columns_available = column_order_field.val().split(',').filter(function (n) {
            return !columnIsVisible(n) && !columnIsLocked(n);
        });

        if (columns_available.length === 0) {
            $('#columns_available_title').hide();
        } else {
            $('#columns_available_title').show();
        }

        display = $('#columns_available');
        display.empty();
        columns_available.sort(compareOptions);
        for (let i = 0; i < columns_available.length; i++) {
            display.append($('<div class="input-group">'
                + '<span id="' + columns_available[i] + '_name" class="form-control input-group-text">' + localized_columns[columns_available[i]] + '</span>'
                + '<button id="' + columns_available[i] + '_add" class="add btn btn-secondary" type="button"><span class="fa fa-plus" aria-hidden="true"></span></button>'
                + '</div>'));
        }
    }

    // Make the newly created "remove/hide column" buttons work
    const removeButtons = $('button.remove');
    removeButtons.unbind('click');
    removeButtons.click(function () {
        const column_index = $(this).attr('id').replace('_remove', '');
        if (columnIsVisible(column_index) && !columnIsLocked(column_index)) {
            setColumnVisibility(column_index, false);
            updateColumnsModal(true, false);
        }
    });

    // Make the newly created "add column" buttons work
    const addButtons = $('button.add, span.add');
    addButtons.unbind('click');
    addButtons.click(function () {
        const column_index = $(this).attr('id').replace('_add', '');
        if (!columnIsVisible(column_index)) {
            setColumnVisibility(column_index, true);
            updateColumnsModal(true, false);
        }
    });

    // Make the newly created "move column up" buttons work
    const upButtons = $('button.up');
    upButtons.unbind('click');
    upButtons.click(function () {
        const column_index = $(this).attr('id').replace('_up', '');
        if (columnIsVisible(column_index)) {
            const column_order_field = $('input#column_order');
            const columns_ordered_all = column_order_field.val().split(',');
            let swap;
            for (let i = 1; i < columns_ordered_all.length; i++) {
                if (columns_ordered_all[i] === column_index) {
                    let offset = 1;
                    while (i - offset > 0 && !columnIsVisible(columns_ordered_all[i - offset])) {
                        offset++; // skip over columns that are not visible
                    }
                    swap = columns_ordered_all[i - offset];
                    columns_ordered_all[i - offset] = columns_ordered_all[i];
                    columns_ordered_all[i] = swap;
                    break;
                }
            }
            column_order_field.val(columns_ordered_all.join(','));
        }
        updateColumnsModal(false, false);
    });

    // Make the newly created "move column down" buttons work
    const downButtons = $('button.down');
    downButtons.unbind('click');
    downButtons.click(function () {
        const column_index = $(this).attr('id').replace('_down', '');
        if (columnIsVisible(column_index)) {
            const column_order_field = $('input#column_order');
            const columns_ordered_all = column_order_field.val().split(',');
            let swap;
            for (let i = 0; i < columns_ordered_all.length - 1; i++) {
                if (columns_ordered_all[i] === column_index) {
                    let offset = 1;
                    while (i + offset < (columns_ordered_all.length - 1) && !columnIsVisible(columns_ordered_all[i + offset])) {
                        offset++; // skip over columns that are not visible
                    }
                    swap = columns_ordered_all[i + offset];
                    columns_ordered_all[i + offset] = columns_ordered_all[i];
                    columns_ordered_all[i] = swap;
                    break;
                }
            }
            column_order_field.val(columns_ordered_all.join(','));
        }
        updateColumnsModal(false, false);
    });
}

function setupColumnsModal() {
    const resetButton = $('#reset_columns');
    resetButton.unbind('click');
    resetButton.click(function() {
        let visibleColumnsField = $('#column_visibility');
        visibleColumnsField.val($('#default_column_visibility').val());

        let columnsOrderField = $('#column_order');
        columnsOrderField.val($('#default_column_order').val());

        updateColumnsModal(true, false);
    });

    $('#columns_modal').on('shown.bs.modal', function () {
        updateColumnsModal(true, true);
    });
}

$(document).ready(function () {
    const inputField = $('#paging_custom_size_input');
    const submitButton = $('#paging_custom_size_submit');
    inputField.on('change paste keyup', function (event) {
        if (event.type === 'keyup' && event.which === 13) {
            $(location).attr('href', $('#paging_custom_size_submit').attr('href'));
        } else {
            $('#paging_custom_size_submit').attr('href',
                $('#pagingsortingtable_js').data('paginglink').replace('999', $(this).val()));
        }
    });

    // Enforce page size limits (Bug 6743)
    inputField.on('focusout', enforcePageSizeLimits);
    submitButton.click(enforcePageSizeLimits);

    // Bug 7329: setup modal for the configuration of the displayed table columns
    setupColumnsModal();

    // Enable bootstrap look.
    $('[data-toggle="tooltip"]').tooltip();
});
