$(document).ready(function () {
    $('[data-bs-toggle="popover"]').popover().click(function () {
        if ($(this).attr('readonly')) {
            $(this).popover('hide').blur();
            return;
        }

        let datePattern;
        let dateFieldPos;
        switch (getLang($(this))) {
            case 'de':
            case 'pl':
                datePattern = /(\d{2})\.(\d{2})\.(\d{4})/;
                dateFieldPos = {d: 1, m: 2, y: 3};
                break;
            case 'fr':
            case 'es':
            case 'pt':
                datePattern = /(\d{2})\/(\d{2})\/(\d{4})/;
                dateFieldPos = {d: 1, m: 2, y: 3};
                break;
            case 'nl':
                datePattern = /(\d{2})-(\d{2})-(\d{4})/;
                dateFieldPos = {d: 1, m: 2, y: 3};
                break;
            default:
                datePattern = /(\d{2})\/(\d{2})\/(\d{4})/;
                dateFieldPos = {d: 2, m: 1, y: 3};
        }

        function dateFromString(dateString, pattern, fieldPos, defaultValue = null) {
            if (dateString) {
                let matcher = pattern.exec(dateString);
                if (matcher) {
                    return new Date(parseInt(matcher[fieldPos.y]), parseInt(matcher[fieldPos.m]) - 1,
                        parseInt(matcher[fieldPos.d]));
                }
            }
            if (defaultValue) {
                defaultValue.setHours(0);
            }
            return defaultValue;
        }

        let selectedDate = dateFromString($(this).val(), datePattern, dateFieldPos, new Date());
        const startDate = dateFromString($(this).data('start-date'), /(\d{4})-(\d{2})-(\d{2})/, {d: 3, m: 2, y: 1});
        if (startDate && selectedDate.getTime() < startDate.getTime()) {
            selectedDate = startDate;
        }
        let endDate = dateFromString($(this).data('end-date'), /(\d{4})-(\d{2})-(\d{2})/, {d: 3, m: 2, y: 1});
        if (endDate && startDate && endDate.getTime() < startDate.getTime()) {
            endDate = startDate;
        }
        if (endDate && selectedDate.getTime() > endDate.getTime()) {
            selectedDate = endDate;
        }
        buildCalendar($(this), selectedDate, startDate, endDate, $(this).data('allow-weekend'));
    });
});

function buildCalendar(textField, selectedDate, startDate, endDate, allowWeekends, monthDelta = 0) {
    const monthDate = new Date(selectedDate);
    monthDate.setMonth(monthDate.getMonth() + monthDelta, 1);
    $('.popover').mousedown(function (event) {
        event.preventDefault();
    });
    const header = createHtmlElement($('.popover-header').html(''), 'div', 'd-flex justify-content-between', null);
    let navButton;
    if (startDate == null ||
        startDate.getFullYear() * 12 + startDate.getMonth() < monthDate.getFullYear() * 12 + monthDate.getMonth()) {
        navButton = createHtmlElement(header, 'a', 'text-primary', [{'role': 'button'}]);
        $(navButton).click(function () {
            buildCalendar(textField, selectedDate, startDate, endDate, allowWeekends, monthDelta - 1);
        });
    } else {
        navButton = createHtmlElement(header, 'a', 'text-secondary', null);
    }
    navButton.html('&laquo;');
    createHtmlElement(header, 'div', null, null).html(
        new Intl.DateTimeFormat(getLang(textField), {month: 'long'}).format(monthDate) + ' ' +
        monthDate.getFullYear());
    if (endDate == null ||
        endDate.getFullYear() * 12 + endDate.getMonth() > monthDate.getFullYear() * 12 + monthDate.getMonth()) {
        navButton = createHtmlElement(header, 'a', 'text-primary', [{'role': 'button'}]);
        $(navButton).click(function () {
            buildCalendar(textField, selectedDate, startDate, endDate, allowWeekends, monthDelta + 1);
        });
    } else {
        navButton = createHtmlElement(header, 'a', 'text-secondary', null);
    }
    navButton.html('&raquo;');

    const content = createHtmlElement($('.popover-body').html(''), 'div', 'd-flex justify-content-center row', null);
    const table = createHtmlElement(content, 'table', null, null);
    const thead = createHtmlElement(table, 'thead', null, null);
    let tr = createHtmlElement(thead, 'tr', null, null);
    let calDate = new Date(monthDate);
    while (calDate.getDay() !== 0) calDate.setDate(calDate.getDate() + 1);
    const df = new Intl.DateTimeFormat($(this).data('lang'), {weekday: 'short'});
    for (let i = 0; i < 7; i++) {
        createHtmlElement(tr, 'th', 'p-2 text-center', null).html(df.format(calDate.setDate(calDate.getDate() + 1)));
    }
    const tbody = createHtmlElement(table, 'tbody', null, null);
    calDate = new Date(monthDate);
    calDate.setDate(0);
    while (calDate.getDay() !== 0) calDate.setDate(calDate.getDate() - 1);
    let td, tdEnabled;

    function cellClass(date, enabled = true) {
        let result = 'p-2 text-center';
        if (date.getTime() === selectedDate.getTime()) {
            result += ' bg-dark text-white';
        } else if (!enabled || date.getMonth() !== monthDate.getMonth()) {
            result += ' text-muted';
        }
        return result;
    }

    while ((calDate.getFullYear() * 12 + calDate.getMonth()) <= (monthDate.getFullYear() * 12 + monthDate.getMonth())) {
        tr = createHtmlElement(tbody, 'tr', null, null);
        for (let i = 0; i < 7; i++) {
            const fieldDate = new Date(calDate.setDate(calDate.getDate() + 1));
            tdEnabled = startDate == null || startDate.getTime() <= fieldDate.getTime();
            tdEnabled &= endDate == null || endDate.getTime() >= fieldDate.getTime();
            tdEnabled &= allowWeekends || (fieldDate.getDay() !== 0 && fieldDate.getDay() !== 6);
            if (tdEnabled) {
                td = createHtmlElement(tr, 'td', cellClass(fieldDate), [{'role': 'button'}]);
                $(td).click(function () {
                    selectDate(textField, fieldDate);
                });
            } else {
                td = createHtmlElement(tr, 'td', cellClass(fieldDate, false), null);
            }
            $(td).html(fieldDate.getDate());
        }
        // if next day is not this month anymore, stop
        if (new Date(calDate.getTime() + 24 * 60 * 60 * 1000).getMonth() > monthDate.getMonth()) {
            break;
        }
    }

    if ((startDate == null || startDate.getTime() <= new Date().getTime()) &&
        (endDate == null) || endDate.getTime() >= new Date().getTime()) {
        const div = createHtmlElement(content, 'div', 'row mt-3', null);
        createHtmlElement(div, 'button', 'btn btn-secondary btn-sm', null).html(textField.data('text-today')).click(function () {
            selectDate(textField, new Date());
        });
    }
}

function getLang(textField) {
    const lang = textField.data('lang');
    if (lang && lang.length > 2) {
        return lang.substring(0, 2);
    } else {
        return lang;
    }
}

function selectDate(textField, date) {
    if (textField.attr('readonly')) {
        textField.popover('hide').blur();
        return;
    }

    let dateString;
    switch (getLang(textField)) {
        case 'de':
        case 'pl':
            dateString = 'd.m.y';
            break;
        case 'fr':
        case 'es':
        case 'pt':
            dateString = 'd/m/y';
            break;
        case 'nl':
            dateString = 'd-m-y';
            break;
        default:
            dateString = 'm/d/y';
    }
    const normalize = function (number, length) {
        let result = '' + number;
        while (result.length < length) {
            result = '0' + result;
        }
        return result;
    };
    dateString = dateString.replace('d', normalize(date.getDate(), 2)).replace(
        'm', normalize(date.getMonth() + 1, 2)).replace('y', normalize(date.getFullYear(), 4));
    $(textField).val(dateString);
    $(textField).popover('hide').blur();
    $(textField).change();
}