/**
 * Holds the information needed to format numbers to localized texts
 * @type {
 *     {decimalSeparator : string},
 *     {groupingSeparator : string},
 *     {amountFormat : string}
 * }
 */
let localInfo;
let loadedLocalInfo = false;

function getLocalInfo() {
    if (!loadedLocalInfo) {
        const lang = $('#blutils-js').attr('data-lang');
        switch (lang) {
            case 'de':
            case 'nl':
                localInfo = {
                    decimalSeparator: ',',
                    groupingSeparator: '.',
                    amountFormat: '$(AMOUNT) $(CURRENCY)'
                };
                break;
            case 'pl':
                localInfo = {
                    decimalSeparator: ',',
                    groupingSeparator: ' ',
                    amountFormat: '$(AMOUNT) $(CURRENCY)'
                };
                break;
            case 'fr':
                localInfo = {
                    decimalSeparator: ',',
                    groupingSeparator: ' ',
                    amountFormat: '$(AMOUNT) $(CURRENCY)'
                };
                break;
            case 'es':
                localInfo = {
                    decimalSeparator: ',',
                    groupingSeparator: '.',
                    amountFormat: '$(CURRENCY)$(AMOUNT)'
                };
                break;
            case 'pt':
                localInfo = {
                    decimalSeparator: ',',
                    groupingSeparator: '.',
                    amountFormat: '$(CURRENCY) $(AMOUNT)'
                };
                break;
            default:
                localInfo = {
                    decimalSeparator: '.',
                    groupingSeparator: ',',
                    amountFormat: '$(CURRENCY)$(AMOUNT)'
                };
        }
        loadedLocalInfo = true;
    }
    return localInfo;
}

/**
 * @deprecated parsing a number as a float will cause precision errors at 14 digits or more (Bug 6799).
 *
 * Converts a float value to a localized string.
 *
 * @param number    the number to convert
 * @param fractions the number of fractions to use
 * @returns {string} the localized string
 */
function localizeNumber(number, fractions) {
    fractions = fractions || 2;
    const exp = Math.pow(10, fractions);
    let result = (Math.round(number * exp) / exp).toString();
    return canonicalToLocalizedNumberString(result, fractions);
}

/**
 * Convert a canonical number String (e.g., "12345.67") to the localized representation (e.g., "12.345,67") using the
 * grouping and decimal separator symbols indicated by {@link getLocalInfo}.
 *
 * @param numberString the localized number string to format
 * @param fractions the number of fractions to use
 * @returns {string} the canonical number string representation
 */
function canonicalToLocalizedNumberString(numberString, fractions) {
    const locale = getLocalInfo();
    let i;
    let result = '' + numberString;

    // check position of decimal separator
    let decimalPos = result.indexOf('.');
    if (decimalPos < 0) {
        result += '.';
        decimalPos = result.length - 1;
    }

    // check number of fractions
    if (decimalPos + fractions + 1 < result.length) {
        result = result.substring(0, decimalPos + fractions + 1);
    } else {
        // append zeros to fill up fractions
        const numberOfZeros = decimalPos + fractions + 1 - result.length;
        for (i = 0; i < numberOfZeros; i++) {
            result += '0';
        }
    }

    // remove last decimal, if fractions in 0
    if (fractions === 0 && result.charAt(result.length - 1) === '.') {
        result = result.substring(0, result.length - 1);
    }

    // replace decimal separator, if different from '.'
    if (locale.decimalSeparator !== '.') {
        result = result.replace('.', locale.decimalSeparator);
    }

    // insert grouping separator each 3 digits
    for (i = decimalPos - 3; i > 0; i -= 3) {
        // don't insert separator before leading '-'
        if (i === 1 && result.charAt(i) === '-') {
            break;
        }

        // build new string with separator at the current position
        result = result.substring(0, i) + locale.groupingSeparator + result.substring(i, result.length);
    }

    return result;
}

/**
 * @deprecated parsing a number as a float will cause precision errors at 14 digits or more (Bug 6799).
 *
 * Converts the localized string into a float variable.
 *
 * @param numberString the string to convert
 * @returns {number} the converted float variable
 */
function parseLocalizedNumber(numberString) {
    return parseFloat(localizedToCanonicalNumberString(numberString));
}

/**
 * Convert a localized number String (e.g., "12.345,67") to the canonical representation (e.g., "12345.67") using the
 * grouping and decimal separator symbols indicated by {@link getLocalInfo}.
 *
 * @param numberString the localized number string to format
 * @returns {string} the canonical number string representation
 */
function localizedToCanonicalNumberString(numberString) {
    let result = numberString;
    const locale = getLocalInfo();

    // grouping?
    if (locale.groupingSeparator != null && locale.groupingSeparator.length > 0) {
        // grouping is used, remove grouping character
        result = result.split(locale.groupingSeparator).join('');
    }

    // decimal separator?
    if (locale.decimalSeparator !== '.') {
        // replace locale decimal separator by '.'
        result = result.split(locale.decimalSeparator).join('.');
    }

    return result;
}

function localizeAmount(number, currency, fraction) {
    number = '' + number;
    let sign;
    if (number.charAt(0) === '-') {
        sign = '-';
        number = number.substring(1);
    } else {
        sign = '';
    }
    return sign + getLocalInfo().amountFormat.replace('$(AMOUNT)',
        canonicalToLocalizedNumberString(number, fraction)).replace('$(CURRENCY)', currency);
}

function amountFieldChanged(amountField, fractions) {
    fractions = fractions || 2;
    if (amountField == null || amountField.length === 0) {
        return;
    }

    let amountString = amountField.val().replace(/[^0-9 ,.'-]/g, '');
    if (amountString != null && amountString.length > 0) {
        const canonicalAmount = localizedToCanonicalNumberString(amountString);
        amountString = canonicalToLocalizedNumberString(canonicalAmount, fractions);
    }
    amountField.val(amountString);
}

/**
 * Runs a dynamic search on the URL for a text field and fills the search result
 * to the search box with the given field ID. The callback function
 * <code>handleLine</code> converts the data to the HTML code that is to be
 * added to the search box.
 *
 * @param textField the text field to do the search for
 * @param searchBox
 *            the search box to add the result lines to
 * @param url
 *            the URL for the search
 * @param resultHandler
 *            the callback function to convert the search result to HTML code
 */
function asyncSearchForTextField(textField, searchBox, url, resultHandler) {
    if (!searchBox.length) {
        return;
    }

    const search = textField.val();
    if (search == null || search.length === 0) {
        searchBox.hide();
        return;
    }

    url = url.replace(/amp;/g, '');
    url = url.replace('SEARCH', encodeURIComponent(search));
    $.ajax({
        url: url,
        mimeType: 'text/plain',
        dataType: 'text',
        success: function (response) {
            let lines;
            if (response == null || response.length === 0) {
                lines = [];
            } else {
                lines = response.split('\n');
            }
            if (lines.length === 0 || lines[0].length === 0) {
                searchBox.hide();
                return;
            }

            const div = document.createElement('DIV');
            $(div).addClass('list-group');
            lines.forEach(function (line) {
                // Bug 6619: HTML-encode the result line to make sure that <script> tags are not executed
                line = htmlEncodeAntiXSS(line);
                const result = resultHandler(line);
                if (result) {
                    // prevent all list items from closing menu on click
                    $(result).mousedown(function (event) {
                        event.preventDefault();
                    });
                    $(div).append(result);
                }
            });

            searchBox.empty();
            searchBox.append(div);
            searchBox.css('zIndex', '1000'); // move to top level
            searchBox.show();
        },
        error: function (message) {
            console.log(message);
            searchBox.hide();
        }
    });
}

/** The currently selected search box item */
let selectedSearchBoxItem = -1;

/**
 * Key event handler for search fields. Allows the user to navigate through the
 * search box using the cursor keys, select an item with ENTER and close the
 * search box with ESC.
 *
 * @param event
 *            the key event to handle
 * @param searchBoxId
 *            the ID of the search box
 * @param searchFunction
 *            the function to execute to update the search
 */
function searchFieldKeyHandler(event, searchBoxId, searchFunction) {
    const searchBoxDiv = $('#' + searchBoxId + ' div');
    if (event.which === 0x1B) {
        // ESC -> close search box
        $('#' + searchBoxId).hide();
    } else if (event.which === 0x28) {
        // down pressed
        selectedSearchBoxItem++;
        if (selectedSearchBoxItem >= searchBoxDiv.children('a').length) {
            selectedSearchBoxItem = -1;
        }

        searchBoxDiv.children('a').each(function () {
            if ($(this).index() === selectedSearchBoxItem) {
                $(this).addClass('active');
            } else {
                $(this).removeClass('active');
            }
        });
    } else if (event.which === 0x26) {
        // up pressed
        if (selectedSearchBoxItem < 0) {
            selectedSearchBoxItem = searchBoxDiv.children('a').length - 1;
        } else {
            selectedSearchBoxItem--;
        }

        searchBoxDiv.children('a').each(function () {
            if ($(this).index() === selectedSearchBoxItem) {
                $(this).addClass('active');
            } else {
                $(this).removeClass('active');
            }
        });
    } else if (event.which === 0x0D) {
        const link = searchBoxDiv.children('a').eq(selectedSearchBoxItem);
        if (link.length) {
            $(link)[0].click();
        }
        selectedSearchBoxItem = -1;
    } else {
        selectedSearchBoxItem = -1;
        searchFunction(event.target);
    }
}

/**
 * Prevents the current document from being submitted when user presses enter to
 * select an item in the search box.
 *
 * @param fieldId
 *            the ID of the search box to add the handler for
 */
function addSearchBoxEnterPrevention(fieldId) {
    $('#' + fieldId).keydown(function (event) {
        if (event.target.id === fieldId && event.which === 0x0D && selectedSearchBoxItem >= 0) {
            event.preventDefault();
        }
    });
}

function setHiddenFormTextParameter(form, value, fieldName) {
    fieldName = fieldName || "text";
    let formId = $(form).attr('id');
    if (formId) {
        formId += '_';
    } else {
        formId = '';
    }
    let input = $(form).find('input[name=' + fieldName + ']');
    if (!input.length) {
        input = createHtmlElement(form, 'INPUT', null, [{
            'type': 'hidden',
            'name': fieldName,
            'id': formId + fieldName
        }]);
    }
    input.val(value);
}

function tableCheckboxSetup(tableId, hasCheckBox) {
    if (hasCheckBox === undefined) {
        hasCheckBox = true;
    }
    return function () {
        let shift = false;
        $(document).on('keydown', function (event) {
            if (event.keyCode === 16) {
                shift = true;
            }
        });
        $(document).on('keyup', function (event) {
            if (event.keyCode === 16) {
                shift = false;
            }
        });

        let lastSelectedIndex = -1;
        $('th[id^=' + tableId + '_head_]').click(function () {
            $(location).attr('href', $(this).data('link'));
        });
        $('input[id^=' + tableId + '_select_][type=checkbox]').click(function () {
            const checkboxId = $(this).attr('id');
            tableCheckBoxChecked(
                checkboxId.substring(checkboxId.length - 3) === 'all', tableId + '_select', checkboxCallback);
        });
        $('#' + tableId + ' tr').each(function (i, e) {
            const checkboxes = $(e).children('td,th').first();
            if (hasCheckBox) {
                $(e).children('td').not(checkboxes).click(function () {
                    const href = $(this).parent().attr("href");
                    if (href) {
                        $(location).attr('href', href);
                    }
                });
                checkboxes.click(function (event) {
                    const checkbox = $(this).children('input[type=checkbox]');
                    if (checkbox.length > 0) {
                        checkbox.click();
                        event.stopPropagation();
                    }
                });
                checkboxes.children('input').click(function (event) {
                    event.stopPropagation();
                    if (shift && lastSelectedIndex >= 0) {
                        shift = false;
                        const selectedIndex = $(this).closest('tr').index();
                        let from = lastSelectedIndex;
                        let to = selectedIndex;
                        if (from > to) {
                            from = to;
                            to = lastSelectedIndex;
                        }
                        $('#' + tableId + ' tr').each(function () {
                            if ($(this).index() > from && $(this).index() < to) {
                                const checkbox = $(this).children('td').first().children('input').first();
                                if (!$(checkbox).prop('checked')) {
                                    $(checkbox).click();
                                }
                            }
                        })
                    } else if ($(this).prop('checked')) {
                        lastSelectedIndex = $(this).closest('tr').index();
                    } else {
                        lastSelectedIndex = -1;
                    }
                });
            } else {
                $(e).children('td').click(function () {
                    const href = $(this).parent().attr("href");
                    if (href) {
                        $(location).attr('href', href);
                    }
                });
            }
        });
    };
}

function formatIban(field, checkIfIban) {
    if (field == null || field.length === 0) {
        return;
    }

    const toCheck = field.val().toUpperCase().replace(/[^A-Z0-9]/g, '');

    if (!checkIfIban || (toCheck.length > 4 && toCheck.length < 35 &&
        toCheck.match(/[A-Z]{2}[0-9]{2}[A-Za-z0-9]{1,30}/) != null)) {
        let formattedIban = toCheck.substring(0, 4);
        for (let i = 4; i < toCheck.length; i += 4) {
            formattedIban += ' ' + toCheck.substring(i, i + 4);
        }
        field.val(formattedIban);
    } else if (!checkIfIban) {
        field.val(toCheck);
    }
}

/**
 * Taken from jquery-ui: the key codes; these should always be allowed in text fields.
 * @type {number[]}
 */
const keyCodes = [8, 46, 40, 35, 13, 27, 36, 37, 34, 33, 39, 9, 38];

/**
 * Checks, if the character in the event is allowed for numeric input text fields. Beside the key codes for controlling
 * the input (backspace, home, cursors etc.) only digits are allowed. If the event has another character, the event is
 * prevented from further execution.
 * @param event the event to check
 */
function checkInputNumeric(event) {
    if (keyCodes.indexOf(event.which) < 0 && (event.key < '0' || event.key > '9') && !event.ctrlKey && !event.metaKey) {
        event.preventDefault();
    }
}

/**
 * Checks, if the character in the event is allowed for decimal input text fields (as amount fields). Beside the key
 * codes for controlling the input (backspace, home, cursors etc.) only digits and the locale decimal and grouping
 * separators are allowed. If the event has another character, the event is prevented from further execution.
 * @param event the event to check
 */
function checkInputDecimalNumeric(event) {
    if (event.key !== getLocalInfo().decimalSeparator && event.key !== getLocalInfo().groupingSeparator) {
        checkInputNumeric(event);
    }
}

/**
 * Checks, if the character in the event is allowed for decimal input text fields (as amount fields). Beside the key
 * codes for controlling the input (backspace, home, cursors etc.) only digits, letters between 'A' and 'Z' (upper and
 * lower case) and space are allowed. If the event has another character, the event is prevented from further execution.
 * @param event the event to check
 */
function checkInputAlphanumeric(event) {
    if (!(event.key >= 'A' && event.key <= 'Z') && !(event.key >= 'a' && event.key <= 'z') && event.key !== ' ') {
        checkInputNumeric(event);
    }
}

function checkInputHexadecimal(event) {
    if (!(event.key >= 'A' && event.key <= 'F') && !(event.key >= 'a' && event.key <= 'f')) {
        checkInputNumeric(event);
    }
}

/**
 * Creates an HTML element of the given type and appends it to the parent.
 *
 * @param parent the element to append the new item to
 * @param name the element type
 * @param clazz the style class of the element
 * @param attrs an array containing the attributes to add to the new element
 * @returns {*|Window.jQuery|HTMLElement} the created HTML element
 */
function createHtmlElement(parent, name, clazz, attrs) {
    const result = $(document.createElement(name));
    if (clazz != null) {
        result.addClass(clazz);
    }
    if (attrs != null) {
        attrs.forEach(function (attr) {
            let sanitizedAttrValue = $("<div/>").text(attrs[attr]).text();
            result.attr(attr, sanitizedAttrValue);
        });
    }
    parent.append(result);
    return result;
}

function replaceAll(where, what, by) {
    if (by === undefined) {
        by = '';
    }
    if (typeof where.replaceAll == 'function') {
        return where.replaceAll(what, by);
    } else {
        let oldVal;
        do {
            oldVal = where;
            where = where.replace(what, by);
        } while (where !== oldVal);
        return where;
    }
}

function htmlEncodeAntiXSS(text) {
    // Bug 6619: HTML-encode dynamic text to make sure that <script> tags are not executed
    return $("<div/>").text(text).html().replaceAll("&amp;", "&");
}

// Bug 7224: Anzeigename am Empfängerfeld in Zahlungsmasken anzeigen
function hideRecipientDisplayName() {
    const displayNameField = $('#recipientNameDescription');
    if (displayNameField.attr('data-recipientname')) {
        // Hide the display name description when recipient data changed
        const nameField = $('#recipientName');
        const accountField = $('#recipientAccount');
        const bankCodeField = $('#recipientBankCode');
        let dataChanged = (nameField.val() !== displayNameField.attr('data-recipientname'))
            || (accountField.length > 0 && accountField.val().replace(/\s+/g, '') !== displayNameField.attr('data-iban').replace(/\s+/g, ''))
            || (bankCodeField.length > 0 && bankCodeField.val() !== displayNameField.attr('data-bic'));
        if (dataChanged) {
            displayNameField.attr('data-recipientname', '');
            displayNameField.attr('data-iban', '');
            displayNameField.attr('data-bic', '');
            displayNameField.hide();
        }
    }
}

/** needed on every page to correct the margin. */
$(function () {
    $(window).resize(function () {
        $('body').css('margin-top', $('nav.fixed-top').outerHeight()).css('margin-bottom', $('footer').outerHeight());
    }).resize();
});

$(document).ready(function () {
    // When window is scrolled, move navbar to background and close its dropdown menus
    window.addEventListener("scroll", function () {
        let pageHasActionmenu = document.getElementsByClassName("bl-action-menu").length > 0;
        let pageHasStickyHeaders = document.getElementsByClassName("sticky-top").length > 0;
        if (pageHasActionmenu || pageHasStickyHeaders) {
            let navbar = $('#top-navbar-container');
            const startFadeoutAt = 30; // reduce navbar opacity when scrollY exceeds this number of pixels
            const hideAt = 70; // hide navbar when scrollY reaches or exceeds this number of pixels
            let opacity = Math.max(0,
                hideAt - startFadeoutAt - Math.max(0, this.scrollY - startFadeoutAt)) / (hideAt - startFadeoutAt);
            if (opacity > 0) {
                // Move navigation bar to the back (lower z-index)
                navbar.show();
                navbar.css("z-index", "100");
                navbar.css("opacity", opacity);
            } else {
                navbar.hide();
            }

            // close dropdown menus
            let dropdownMenus = navbar.find('.dropdown-menu');
            for (let i = 0; i < dropdownMenus.length; i++) {
                dropdownMenus[i].classList.remove("show");
            }

            // update dropdown button properties
            let dropdownButtons = navbar.find('.dropdown-toggle');
            dropdownButtons.attr("aria-expanded", "false");
            for (let i = 0; i < dropdownButtons.length; i++) {
                dropdownButtons[i].classList.remove("show");
            }
        }
    });

    // When user interacts with navbar, pull it back to the front (use standard z-index for class 'fixed-top')
    const navbarContainer = $('#top-navbar-container');
    navbarContainer.click(function () {
        $(this).css("z-index", "");
        $(this).css("opacity", "1.0");
    });
    navbarContainer.keydown(function () {
        $(this).css("z-index", "");
        $(this).css("opacity", "1.0");
    });
});
