/**
 * Parameters for the current script
 * @type {
 *     {initializeAccountAndOriginator : Boolean},
 *     {hasError: Boolean},
 *     {isNew : Boolean},
 *     {overwriteRecipientData : Boolean},
 *     {showRecipientWithAccountNumber : Boolean},
 *     {searchOriginatorsUrl : String},
 *     {searchRecipientsUrl : String},
 *     {searchOriginatorCustomerNumberUrl : String},
 *     {accounts : [
 *         {accountId : number},
 *         {currency : String},
 *         {holder : String},
 *         {defaultOriginator: String},
 *         {tenantId : number}
 *     ]},
 *     {originators : [
 *         {id : number},
 *         {tenantId : number},
 *         {defaultAccountId : number},
 *         {name : String},
 *         {name2 : String},
 *         {street : String},
 *         {town : String},
 *         {contact : String}
 *     ]},
 *     {recipients : [
 *         {id : number},
 *         {tenantId : number}
 *     ]},
 *     {displayNameLocalized : String}
 * }
 */
const dtazvpaymentEditableParams = JSON.parse(atob($('#dtazvpayment_editable_js').attr('data-params')));

// these booleans will be used for initialization of accounts and originators
let accountHasChanged = false;
let accountIsChangingByOriginator = false;
let originatorIsChangingByAccount = false;
let originatorHasChanged = false;

// flag if select#originatorName or input#originatorName is used
let useSelect = false;

//this function selects the default account for a selected originator
function selectDefaultAccount(name) {
    if (dtazvpaymentEditableParams.originators != null) {
        dtazvpaymentEditableParams.originators.forEach(function (originator) {
            if (name === originator.name) {
                if (originator.defaultAccountId) {
                    accountIsChangingByOriginator = true;
                    $('#originatorAccountId').val(originator.defaultAccountId).change();
                }
            }
        });
    }
}

//this function selects the default originator or account holder, if nothing is set "" will be set for a selected account
function selectDefaultOriginator(id) {
    if (dtazvpaymentEditableParams.accounts != null) {
        let originatorToSelect = "";
        dtazvpaymentEditableParams.accounts.forEach(function (account) {
            if (id === account.accountId) {
                if (account.holder) {
                    originatorToSelect = account.holder;
                }
                if (account.defaultOriginator) {
                    originatorToSelect = account.defaultOriginator;
                }
                originatorIsChangingByAccount = true;
                if (!useSelect) {
                    $('input#originatorName').val(originatorToSelect).change();
                } else {
                    $('select#originatorName').val(originatorToSelect).change();
                }
            }
        });
    }
}

function toggleOriginators(tenantId) {
    if (tenantId < 0 || !dtazvpaymentEditableParams.originators) {
        // if no tenant can be found in account there is no need to show and hide some originators
        return;
    }

    const originatorSelect = $('select#originatorName');
    if (!originatorSelect.length) {
        dtazvpaymentEditableParams.originators.forEach(function (originator) {
            if (originator.tenantId === tenantId) {
                $('#select_originator_item_' + originator.id).show();
            } else {
                $('#select_originator_item_' + originator.id).hide();
            }
        });
    } else {
        const selectedOriginator = originatorSelect.val();
        if (originatorSelect.data('originalhtml') == null) {
            originatorSelect.data('originalhtml', originatorSelect.html());
        } else {
            originatorSelect.html(originatorSelect.data('originalhtml'));
        }
        let originatorId;
        let foundOriginator;
        originatorSelect.find('OPTION').each(function () {
            originatorId = parseInt($(this).val());
            foundOriginator = false;
            dtazvpaymentEditableParams.originators.forEach(function (originator) {
                foundOriginator |= (originator.id === originatorId && originator.tenantId === tenantId);
            });
            if (!foundOriginator) {
                $(this).remove();
            }
        });
        originatorSelect.val(selectedOriginator);
        if (!originatorSelect.val()) {
            //clear fields - tenant does not contain this originator
            updateOriginator("", "", "", "", "");
        }
    }
}

function originatorAccountChanged() {
    if (dtazvpaymentEditableParams.initializeAccountAndOriginator) {
        const accountId = parseInt($('#originatorAccountId').val());

        const currencyField = $('#currency');
        const amountField = $('#amount');
        if (dtazvpaymentEditableParams.accounts != null) {
            dtazvpaymentEditableParams.accounts.forEach(function (account) {
                if (account.accountId === accountId) {
                    if (currencyField.length && (amountField.val() == null || amountField.val() === '')) {
                        currencyField.val(account.currency);
                    }

                    if (account.tenantId) {
                        toggleOriginators(account.tenantId);
                        toggleFolders(account.tenantId);
                    }
                }
            });
        }
        fillOriginatorCustomerNumberFromOriginatorAccountId();
        updateToolTipAccountId();
        accountHasChanged = true;

        if (!originatorHasChanged) {
            selectDefaultOriginator(accountId);
        } else {
            if (accountIsChangingByOriginator) {
                accountHasChanged = false;
                accountIsChangingByOriginator = false;
            }
        }
    }
}

function originatorSelected() {
    if (dtazvpaymentEditableParams.initializeAccountAndOriginator) {
        if (dtazvpaymentEditableParams.originators == null) {
            return;
        }

        let compareTo;
        if (useSelect) {
            compareTo = $('select#originatorName').val();
        } else {
            compareTo = $('input#originatorName').val();
        }
        let foundOriginator = false;
        dtazvpaymentEditableParams.originators.forEach(function (originator) {
            let comparison;
            if (useSelect) {
                comparison = compareTo === '' + originator.id;
            } else {
                comparison = compareTo === '' + originator.name;
            }
            if (comparison) {
                foundOriginator = true;
                originatorHasChanged = true;
                updateOriginator(originator.name, originator.name2, originator.street, originator.town, originator.contact);
                if (!accountHasChanged) {
                    selectDefaultAccount(originator.name);
                } else {
                    if (originatorIsChangingByAccount) {
                        //reset boolean - originator is only changed because account is changed
                        originatorHasChanged = false;
                        originatorIsChangingByAccount = false;
                    }
                }
            }
        });
        if (!foundOriginator) {
            updateOriginator(compareTo, "", "", "", "");
        }
    }
}

function updateOriginator(name, name2, street, town, contact) {
    $('input#originatorName').val(name);
    $('#originatorName2').val(name2);
    $('#originatorStreet').val(street);
    $('#originatorTown').val(town);
    $('#originatorContact').val(contact);

    $('#originator_search').hide();
}

function getTenant() {
    let tenantId = -1;
    const accountId = parseInt($('#originatorAccountId').val());
    dtazvpaymentEditableParams.accounts.forEach(function (account) {
        if (account.tenantId != null && account.accountId === accountId) {
            tenantId = account.tenantId;
        }
    });
    return tenantId;
}

function searchOriginators() {
    asyncSearchForTextField($('input#originatorName'), $('#originator_search'),
        dtazvpaymentEditableParams.searchOriginatorsUrl.replace('999', '' + getTenant()),
        function (line) {
            const lineParts = line.split('\t');
            if (lineParts.length < 6) {
                return '';
            }

            const result = $('<A/>').prop('role', 'button').addClass('list-group-item');
            $(result).click(function () {
                updateOriginator(lineParts[1], lineParts[2], lineParts[3], lineParts[4], lineParts[5]);
                originatorHasChanged = true;
                if (dtazvpaymentEditableParams.initializeAccountAndOriginator && !accountHasChanged) {
                    selectDefaultAccount(name);
                } else {
                    if (originatorIsChangingByAccount) {
                        //reset boolean - originator is only changed because account is changed
                        originatorHasChanged = false;
                        originatorIsChangingByAccount = false;
                    }
                }
            });
            $(result).html(lineParts[1]);
            return result;
        });
}

function updateRecipient(recipientId, displayName, name, iban, bic, name2, street, town, country, bankName, bankName2,
                         bankStreet, bankTown, bankCountry, purp1, purp2, purp3, purp4, fee) {
    $('#recipientId').val(recipientId);
    $('#recipientName').val(name);
    if (dtazvpaymentEditableParams.overwriteRecipientData) {
        $('#recipientAccount').val(iban).change();
        $('#recipientBankCode').val(bic).change();
        $('#recipientName2').val(name2);
        $('#recipientStreet').val(street);
        $('#recipientTown').val(town);
        if (country != null && country !== '') {
            $('#recipientCountryCode').val(country);
            $('#recipientCountryCode_hidden').val(country);
        }
        $('#recipientBankName').val(bankName);
        $('#recipientBankName2').val(bankName2);
        $('#recipientBankStreet').val(bankStreet);
        $('#recipientBankTown').val(bankTown);
        if (bankCountry != null && bankCountry !== '') {
            $('#recipientBankCountryCode').val(bankCountry);
            $('#recipientBankCountryCode_hidden').val(bankCountry);
        }
        $('#purpose').val(purp1);
        $('#purpose2').val(purp2);
        $('#purpose3').val(purp3);
        $('#purpose4').val(purp4);

        $('#feeType').val(fee);
    }
    $('#recipient_search').hide();

    const displayNameField = $('#recipientNameDescription');
    if (displayName != null && displayName.length > 0) {
        // remember the data of the selected recipient, so we know when to hide the display name field again
        displayNameField.attr('data-recipientname', name);
        if (iban != null) {
            displayNameField.attr('data-iban', iban);
        }
        if (bic != null) {
            displayNameField.attr('data-bic', bic);
        }

        // show display name
        let prefixedDisplayName = dtazvpaymentEditableParams.displayNameLocalized + ": " + displayName;
        displayNameField.html(prefixedDisplayName);
        displayNameField.removeClass("hidden");
        displayNameField.show();
    } else {
        displayNameField.hide();
    }
}

function searchRecipients() {
    asyncSearchForTextField($('#recipientName'), $('#recipient_search'),
        dtazvpaymentEditableParams.searchRecipientsUrl.replace('999', '' + getTenant()),
        function (line) {
            const lineParts = line.split('\t');
            if (lineParts.length < 2) {
                return '';
            }

            // 0:databaseId, 1:paymentType, 2:displayName, 3:name, 4:accountNumber, 5:bankCode, 6:name2,
            // 7:street, 8:town, 9:country, 10:bankName, 11:bankName2, 12:bankStreet, 13:bankTown, 14:bankCountry,
            // 15:purpose1, 16:purpose2, 17:purpose3, 18:purpose4, 19:fee
            let databaseId = lineParts[0];
            let paymentType = lineParts[1];
            let displayName = lineParts[2];
            let name = lineParts[3];
            let accountNumber = lineParts[4];
            let bankCode = lineParts[5];
            let name2 = lineParts[6];
            let street = lineParts[7];
            let town = lineParts[8];
            let country = lineParts[9];
            let bankName = lineParts[10];
            let bankName2 = lineParts[11];
            let bankStreet = lineParts[12];
            let bankTown = lineParts[13];
            let bankCountry = lineParts[14];
            let purpose1 = lineParts[15];
            let purpose2 = lineParts[16];
            let purpose3 = lineParts[17];
            let purpose4 = lineParts[18];
            let fee = lineParts[19];

            const result = $('<A/>').prop('role', 'button').addClass('list-group-item');
            $(result).click(function () {
                updateRecipient(databaseId, displayName, name, accountNumber, bankCode,
                    name2, street, town, country, bankName, bankName2, bankStreet,
                    bankTown, bankCountry, purpose1, purpose2, purpose3, purpose4, fee);
            });
            $(result).html(displayName);
            if (dtazvpaymentEditableParams.showRecipientWithAccountNumber) {
                if (accountNumber != null && accountNumber !== '') {
                    result.append(' (');
                    if (bankCode != null && bankCode !== '') {
                        result.append(bankCode + '/');
                    }
                    result.append(accountNumber + ')');
                }
            }
            return result;
        });
}

function isEURAccount() {
    let isEURAccount = false;
    const accountId = parseInt($('#originatorAccountId').val());
    dtazvpaymentEditableParams.accounts.forEach(function (account) {
        if (account.accountId === accountId && account.currency != null) {
            isEURAccount = account.currency === 'EUR';
        }
    });
    return isEURAccount;
}

function amountCurrencyChanged() {
    const currencyField = $('#currency');
    if (currencyField.length === 0) {
        return;
    }
    // Bug 6172 - Gegenwertzahlung bei DTAZV nur für EUR Konten erlauben
    // check if originator account has currency EUR
    const counterValueField = $('#counterValue');
    if (isEURAccount()) {
        if (currencyField.val() === 'EUR') {
            counterValueField.prop('disabled', false);
        } else {
            counterValueField.prop('disabled', true);
        }
    } else {
        counterValueField.prop('checked', false);
        counterValueField.prop('disabled', true);
        $('#counterValueCurrency').prop('disabled', true);
        currencyField.prop('disabled', false);
    }
    getConvertedAmountText();
}

function fillOriginatorCustomerNumberFromOriginatorAccountId() {
    const originatorAccountId = $('#originatorAccountId').val();
    if (originatorAccountId != null && originatorAccountId !== '') {
        const accountId = parseInt(originatorAccountId);
        const link = dtazvpaymentEditableParams.searchOriginatorCustomerNumberUrl.replace('999', '' + accountId);
        $.ajax({
            url: link,
            mimeType: 'text/plain',
            dataType: 'text',
            success: function (result) {
                $('#originatorCustomerNumber').val(result.toString());
            }
        });
    }
}

$(document).ready(function () {
    addSearchBoxEnterPrevention('originatorName');
    addSearchBoxEnterPrevention('recipientName');

    const originatorAccountField = $('#originatorAccountId');
    originatorAccountField.change(originatorAccountChanged);
    $('input#originatorName').keyup(function (event) {
        searchFieldKeyHandler(event, 'originator_search', searchOriginators);
    }).focusout(function () {
        $('#originator_search').hide();
    }).change(originatorSelected);
    $('a[id^=select_originator_link_]').click(function () {
        updateOriginator(
            $(this).data('name'), $(this).data('name2'), $(this).data('street'), $(this).data('town'),
            $(this).data('contact'));
        originatorHasChanged = true;
        if (dtazvpaymentEditableParams.initializeAccountAndOriginator && !accountHasChanged) {
            selectDefaultAccount($(this).data('name'));
        } else {
            if (originatorIsChangingByAccount) {
                //reset boolean - originator is only changed because account is changed
                originatorHasChanged = false;
                originatorIsChangingByAccount = false;
            }
        }
    });
    // if new originators are not allowed there is a select with id=originatorName
    const originatorNameSelect = $('select#originatorName');
    if (originatorNameSelect.length) {
        useSelect = true;
        originatorNameSelect.change(originatorSelected);
    } else {
        useSelect = false;
    }

    $('#recipientName').keyup(function (event) {
        searchFieldKeyHandler(event, 'recipient_search', searchRecipients);
    }).focusout(function () {
        $('#recipient_search').hide();
    }).change(function () {
        hideRecipientDisplayName();
    });
    $('a[id^=select_recipient_link_]').click(function () {
        let id = $(this).attr('id').substring('select_recipient_link_'.length);
        updateRecipient(
            id, $(this).data('displayname'), $(this).data('name'), $(this).data('accountnumber'), $(this).data('bankcode'),
            $(this).data('name2'), $(this).data('street'), $(this).data('town'), $(this).data('country'),
            $(this).data('bankname'), $(this).data('bankname2'), $(this).data('bankstreet'), $(this).data('banktown'),
            $(this).data('bankcountry'), $(this).data('purpose1'), $(this).data('purpose2'), $(this).data('purpose3'),
            $(this).data('purpose4'), $(this).data('feetype'));
    });
    const currencyField = $('#currency');
    if (currencyField.length) {
        $('#amount').change(getConvertedAmountText);
        currencyField.change(amountCurrencyChanged);
        $('#counterValue').change(counterValueChanged);
        $('#counterValueCurrency').change(getConvertedAmountText);
        originatorAccountField.change(amountCurrencyChanged);
    }

    postFormatAmountOperation = getConvertedAmountText;
    const recipientAccountField = $('#recipientAccount');
    recipientAccountField.change(function () {
        formatIban($(this), true);
        hideRecipientDisplayName();
    }).keydown(checkInputDtazv);

    const recipientBankCodeField = $('#recipientBankCode');
    recipientBankCodeField.change(function () {
        hideRecipientDisplayName();
    });

    if (!dtazvpaymentEditableParams.hasError) {
        if (dtazvpaymentEditableParams.isNew) {
            //for a new open payment the last account and the last default originator will be set.
            // check if an originator data was set, if not set the default originator/ account holder or "" and fill originatorCustomerNumber
            originatorAccountChanged();
        }
    }
    const tenantId = getTenant();
    if (tenantId) {
        toggleOriginators(tenantId);
        toggleFolders(tenantId);
    }

    // reset booleans
    originatorHasChanged = false;
    accountHasChanged = false;
    accountIsChangingByOriginator = false;
    originatorIsChangingByAccount = false;

    //mark amount
    $('#amount_field').find('label.control-label').addClass("mandatorylabel");
});
